<?php
namespace Bitrix\Report;

use Bitrix\Main\Loader;
use Bitrix\Socialnetwork\UserToGroupTable;
use CPHPCache;

class RightsManager
{
	const ACCESS_READ = 'access_read';
	const ACCESS_EDIT = 'access_edit';
	const ACCESS_FULL = 'access_full';

	protected $userId;

	public function __construct($userId)
	{
		$this->userId = intval($userId);
	}

	/**
	 * Checks the right to read.
	 * @param int $reportId Id report.
	 * @return bool
	 */
	public function canRead($reportId)
	{
		if(!$this->isOwner($reportId))
		{
			$listSharing = Sharing::getEntityOfSharing($reportId);
			$listEntity = $this->getGroupsAndDepartments();
			foreach($listSharing as $sharingRow)
			{
				if(in_array($sharingRow['ENTITY'], $listEntity))
				{
					if($this->compareAccess($sharingRow['RIGHTS'], self::ACCESS_READ) >= 0)
						return true;
				}
			}
			return false;
		}
		return true;
	}

	/**
	 * Checks the right to edit.
	 * @param int $reportId Id report.
	 * @return bool
	 */
	public function canEdit($reportId)
	{
		if(!$this->isOwner($reportId))
		{
			$listSharing = Sharing::getEntityOfSharing($reportId);
			$listEntity = $this->getGroupsAndDepartments();
			foreach($listSharing as $sharingRow)
			{
				if(in_array($sharingRow['ENTITY'], $listEntity))
				{
					if($this->compareAccess($sharingRow['RIGHTS'], self::ACCESS_EDIT) >= 0)
						return true;
				}
			}
			return false;
		}
		return true;
	}

	/**
	 * Checks the right to delete.
	 * @param int $reportId Id report.
	 * @return bool
	 */
	public function canDelete($reportId)
	{
		return $this->isOwner($reportId);
	}

	/**
	 * Checks the right to share.
	 * @param int $reportId Id report.
	 * @return bool
	 */
	public function canShare($reportId)
	{
		return $this->isOwner($reportId);
	}

	/**
	 * Returns an array of user groups and departments.
	 * @return array
	 * @throws \Bitrix\Main\ArgumentException
	 * @throws \Bitrix\Main\LoaderException
	 */
	public function getGroupsAndDepartments()
	{
		$cacheTime = defined('BX_COMP_MANAGED_CACHE') ? 3153600 : 3600*4;
		$cacheId = 'report-sharing-rights-'.$this->userId;
		$cacheDir = '/report/sharing/rights/'.$this->userId;
		$cache = new CPHPCache;
		if($cache->initCache($cacheTime, $cacheId, $cacheDir))
		{
			$listEntity = $cache->getVars();
		}
		else
		{
			global $CACHE_MANAGER;
			$cache->startDataCache();
			$listEntity = array(Sharing::CODE_USER.$this->userId);
			$userObject = \CUser::getByID($this->userId);
			if($userData = $userObject->fetch())
			{
				if(Loader::includeModule('socialnetwork'))
				{
					$queryObject = UserToGroupTable::getList(array(
						'select' => array('GROUP_ID'),
						'filter' => array(
							'USER_ID' => $userData['ID'],
							'ROLE' => array(
								UserToGroupTable::ROLE_USER,
								UserToGroupTable::ROLE_MODERATOR,
								UserToGroupTable::ROLE_OWNER
							)
						)
					));
					while ($groupData = $queryObject->fetch())
						$listEntity[] = Sharing::CODE_SOCNET_GROUP.$groupData['GROUP_ID'];
				}

				if(!empty($userData['UF_DEPARTMENT']) && Loader::includeModule('iblock'))
				{
					foreach($userData['UF_DEPARTMENT'] as $departmentId)
					{
						$res = \CIBlockSection::GetNavChain(0, $departmentId);
						while ($row = $res->Fetch())
						{
							$listEntity[] = Sharing::CODE_DEPARTMENT . $row['ID'];
						}
					}
				}
			}
			$CACHE_MANAGER->startTagCache($cacheDir);
			$CACHE_MANAGER->registerTag("sonet_user2group_U".$this->userId);
			$CACHE_MANAGER->registerTag("USER_CARD_".intval($this->userId/TAGGED_user_card_size));
			$CACHE_MANAGER->endTagCache();
			$cache->endDataCache($listEntity);
		}

		return $listEntity;
	}

	private function isOwner($reportId)
	{
		$reportId = intval($reportId);

		$hasReport = ReportTable::getCount(
			array('=ID' => $reportId, '=CREATED_BY' => $this->userId));

		if($hasReport)
			return true;
		else
			return false;
	}

	/**
	 * @param $access1
	 * @param $access2
	 * @return int Returns < 0 if $access1 is less than $access2; > 0
	 *    if $access1 is greater than $access2, and 0 if they are equal.
	 * @internal
	 */
	private function compareAccess($access1, $access2)
	{
		switch($access1)
		{
			case 'access_read':
				self::ACCESS_READ;
				$access1Pos = 2;
				break;
			case 'access_edit':
				self::ACCESS_EDIT;
				$access1Pos = 3;
				break;
			case 'access_full':
				self::ACCESS_FULL;
				$access1Pos = 4;
				break;
			default:
				$access1Pos = -1;
		}
		switch($access2)
		{
			case 'access_read':
				self::ACCESS_READ;
				$access2Pos = 2;
				break;
			case 'access_edit':
				self::ACCESS_EDIT;
				$access2Pos = 3;
				break;
			case 'access_full':
				self::ACCESS_FULL;
				$access2Pos = 4;
				break;
			default:
				$access2Pos = -1;
		}

		if($access1Pos == $access2Pos)
			return 0;

		return $access1Pos > $access2Pos? 1 : -1;
	}
}